(*
**	Copyright (C) 2006-2008  bCODE Pty Ltd (www.bcode.com)
**	Written and maintained by Erik de Castro Lopo <erikd@mega-nerd.com>
**
**	This program is free software: you can redistribute it and/or modify
**	it under the terms of the GNU General Public License as published by
**	the Free Software Foundation, either version 3 of the License, or
**	(at your option) any later version.
**
**	This program is distributed in the hope that it will be useful,
**	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
**	You should have received a copy of the GNU General Public License
**	along with this program.  If not, see <http://www.gnu.org/licenses/>.
*)

open ExtString


type t =
{	swap_xy : bool ;
	x_gain   : float ;
	x_offset : float ;
	y_gain   : float ;
	y_offset : float ;

	top_lines : string list ;
	bottom_lines : string list ;
	}

let empty_config =
	{	swap_xy = false ;
		x_gain = 0.5 ; x_offset = 0.0 ;
		y_gain = 0.5 ; y_offset = 0.0 ;
		top_lines = [] ; bottom_lines = []
		}


let parse_config lines =
	let safe_find haystack needle =
		try
			String.find haystack needle
		with Invalid_string ->
			-1
	in
	let parse (config, start) line =
		try
			if safe_find line "Screen size" > 0 then
				config, start
			else if line.[0] == '#' then
				raise Invalid_string
			else
			let start, rest = String.split line ":" in
			match String.strip start with
			|	"swap_xy" ->
					{ config with swap_xy = bool_of_string (String.strip rest) }, false
			|	"x_gain" ->
					{ config with x_gain = float_of_string rest }, false
			|	"x_offset" ->
					{ config with x_offset = float_of_string rest }, false
			|	"y_gain" ->
					{ config with y_gain = float_of_string rest }, false
			|	"y_offset" ->
					{ config with y_offset = float_of_string rest }, false
			|	_ -> raise Invalid_string
		with
		|	Failure "float_of_string" ->
				failwith (Printf.sprintf "float_of_string failed on '%s'." line)
		|	Invalid_string ->
			if start then
				{ config with top_lines = line :: config.top_lines }, true
			else
				{ config with bottom_lines = line :: config.bottom_lines }, false
		|	Invalid_argument "bool_of_string" ->
				failwith (Printf.sprintf "bool_of_string failed on '%s'." line)
	in
	let x = fst (List.fold_left parse (empty_config, true) lines) in
	{ x with top_lines = List.rev x.top_lines ; bottom_lines = List.rev x.bottom_lines ; }


let read filename =
	try
		let lines = Util.read_file_lines filename in
		parse_config lines
	with Sys_error _ ->
		let dir, fname = Util.file_path_split filename in
		Util.rec_mkdir dir ;
		empty_config

let write config filename =
	let data =
		config.top_lines
		@
		(	Printf.sprintf
					"\n\
					swap_xy  : %s\n\
					\n\
					x_gain   : % f\n\
					x_offset : % f\n\
					y_gain   :  %f\n\
					y_offset : % f\n\
					\n"
					(string_of_bool config.swap_xy)
					config.x_gain config.x_offset config.y_gain config.y_offset
			::
			config.bottom_lines
			)
	in
	Util.write_whole_file ((String.join "\n" data) ^ "\n\n") filename
