(*
**	Copyright (C) 2008  bCODE Pty Ltd (www.bcode.com)
**	Written and maintained by Erik de Castro Lopo <erikd@mega-nerd.com>
**
**	This program is free software: you can redistribute it and/or modify
**	it under the terms of the GNU General Public License as published by
**	the Free Software Foundation, either version 3 of the License, or
**	(at your option) any later version.
**
**	This program is distributed in the hope that it will be useful,
**	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
**	You should have received a copy of the GNU General Public License
**	along with this program.  If not, see <http://www.gnu.org/licenses/>.
*)

type t =
	(*
	**	This is currently the only permutations supported.
	**	More permutations can be added by adding a name here and
	**	extending the function build_permutations below.
	*)
	|	Normal
	|	Flip_lr_flip_ud
	|	Swap_xy_flip_lr


let debug = false

let swap_x_y c =
	{ Complex.re = c.Complex.im ; im = c.Complex.re }


let normalize_point_array ca =
	let min_max (min_x, max_x, min_y, max_y) c =
		let x, y = c.Complex.re, c.Complex.im in
		min min_x x, max max_x x, min min_y y, max max_y y
	in
	let start = ca.(0).Complex.re, ca.(0).Complex.re, ca.(0).Complex.im, ca.(0).Complex.im in
	let min_x, max_x, min_y, max_y = Array.fold_left min_max start ca in
	let diff_x, diff_y = max_x -. min_x, max_y -. min_y in
	let norm c =
		let x, y = c.Complex.re, c.Complex.im in
		{ Complex.re = (x -. min_x) /. diff_x ; im = (y -. min_y) /. diff_y }
	in
	Array.map norm ca


let build_permutations a =
	let na = normalize_point_array a in
	[|	Normal, na ;
		Flip_lr_flip_ud, [| na.(3) ; na.(2) ; na.(1) ; na.(0) |] ;
		Swap_xy_flip_lr, Array.map swap_x_y [| na.(1) ; na.(0) ; na.(3) ; na.(2) |] ;
		|]


let string_of_orientation o =
	match o with
	|	Normal -> "Normal"
	|	Flip_lr_flip_ud -> "Flip_lr_flip_ud"
	|	Swap_xy_flip_lr -> "Swap_xy_flip_lr"


let str_of_cmplx_array a =
	let clist = Array.to_list a in
	let slist = List.map (fun c -> Printf.sprintf "(%4.2f, %4.2f) " c.Complex.re c.Complex.im) clist in
	"[| " ^ (String.concat " " slist) ^ " |]"


let distance_of_pair a b =
	Complex.norm (Complex.sub a b)

let max_distance a b =
	assert (Array.length a == Array.length b) ;
	Array.fold_left (fun x y -> max x y) 0.0 (Array.mapi (fun i x -> distance_of_pair x b.(i)) a)

let find_orientation cross_perms clicks =
	let bestest (ora, dista) (ob, points) =
		let dist = max_distance clicks points in
		if debug then
			Printf.printf "  %-20s :\n    %s\n    %s\n    dist = %f\n" (string_of_orientation ob) (str_of_cmplx_array clicks) (str_of_cmplx_array points) dist ;
		if dist < dista then ob, dist else ora, dista
	in
	let first_o = fst cross_perms.(0) in
	let first_d = max_distance clicks (snd cross_perms.(0)) in
	let orient, _ = Array.fold_left bestest (first_o, first_d) cross_perms in
	orient


let find crosses clicks =
	let permutations = build_permutations crosses in
	let normalized_clicks = normalize_point_array clicks in
	find_orientation permutations normalized_clicks


let update_config calibrate_gain swap_xy config crosses clicks =
	if debug then
	(	Printf.printf "crosses : [| " ;
		Array.iter (fun x -> Printf.printf "(% 4.0f, % 4.0f) " x.Complex.re x.Complex.im) crosses ;
		Printf.printf "|]\nclicks  : [| " ;
		Array.iter (fun x -> Printf.printf "(% 4.0f, % 4.0f) " x.Complex.re x.Complex.im) clicks ;
		print_endline "|]" ;
		) ;

	(* First calculate the X direction correction factor. *)
	let x_left_click = 0.5 *. (clicks.(0).Complex.re +. clicks.(2).Complex.re) /. calibrate_gain in
	let x_right_click = 0.5 *. (clicks.(1).Complex.re +. clicks.(3).Complex.re) /. calibrate_gain in
	let x_click_dist = x_right_click -. x_left_click in

	let x_left_cross = 0.5 *. (crosses.(0).Complex.re +. crosses.(2).Complex.re) in
	let x_right_cross = 0.5 *. (crosses.(1).Complex.re +. crosses.(3).Complex.re) in
	let x_cross_dist = x_right_cross -. x_left_cross in

	let x_gain = x_cross_dist /. x_click_dist in
	let x_offset = x_left_cross -. x_gain *. x_left_click in

	(* Now calculate the Y direction correction factor. *)
	let y_top_click = 0.5 *. (clicks.(0).Complex.im +. clicks.(1).Complex.im) /. calibrate_gain in
	let y_bottom_click = 0.5 *. (clicks.(2).Complex.im +. clicks.(3).Complex.im) /. calibrate_gain in
	let y_click_dist = y_bottom_click -. y_top_click in

	let y_top_cross = 0.5 *. (crosses.(0).Complex.im +. crosses.(1).Complex.im) in
	let y_bottom_cross = 0.5 *. (crosses.(2).Complex.im +. crosses.(3).Complex.im) in
	let y_cross_dist = y_bottom_cross -. y_top_cross in

	let y_gain = y_cross_dist /. y_click_dist in
	let y_offset = y_top_cross -. y_gain *. y_top_click in
	{ config with
			Config.swap_xy = swap_xy ;
			x_gain = x_gain  ; x_offset = x_offset ;
			y_gain = y_gain ; y_offset = y_offset ;
			}


let calc_config calibrate_gain config crosses clicks =
	let orientation = find crosses clicks in
	if debug then
		print_endline (string_of_orientation orientation) ;
	match orientation with
	|	Normal ->
			update_config calibrate_gain false config crosses clicks
	|	Flip_lr_flip_ud ->
			update_config calibrate_gain false config crosses clicks
	|	Swap_xy_flip_lr ->
			update_config calibrate_gain true config crosses (Array.map swap_x_y clicks)

