(*
**	Copyright (C) 2006-2008  bCODE Pty Ltd (www.bcode.com)
**	Written and maintained by Erik de Castro Lopo <erikd@mega-nerd.com>
**
**	This program is free software: you can redistribute it and/or modify
**	it under the terms of the GNU General Public License as published by
**	the Free Software Foundation, either version 3 of the License, or
**	(at your option) any later version.
**
**	This program is distributed in the hope that it will be useful,
**	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
**	You should have received a copy of the GNU General Public License
**	along with this program.  If not, see <http://www.gnu.org/licenses/>.
*)

open ExtString

type cross_t =
	|	White_cross
	|	Black_cross

type four_point_t =
{	top_left : Complex.t ;
	top_right : Complex.t ;
	bottom_left : Complex.t ;
	bottom_right : Complex.t ;
	}

type gui_t =
{	win : GWindow.window ;
	darea : GMisc.drawing_area ;
	crosses : Complex.t array ;
	clicks : Complex.t array ;
	mutable index : int ;
	mutable last_press : Complex.t ;
	mutable complete : bool ;
	}


(*
**	Although we assume the touchscreen has a resolution of 1024x768, we
**	cannot be sure that the screen has the same resolution. If the
**	touchscreen returns a value outside of what the screen can handle then
**	the X server (or Gdk toolkit) clips the value to lie inside the screen
**	dimensions.
**
**	To fix this, during calibration we scale X and Y so they are guaranteeed
**	to be inside the screen resolution.
*)
let touchscreen_width = 1024.0
let touchscreen_height = 768.0

let get_screen_size () =
	let width = Gdk.Screen.width () in
	let height = Gdk.Screen.height () in
	(width, height)

let do_cross cr x y clen =
	let two_clen = 2.0 *. clen in
	Cairo.move_to cr (x -. clen) (y -. clen) ;
	Cairo.rel_line_to cr two_clen two_clen ;
	Cairo.stroke cr ;
	Cairo.move_to cr (x -. clen) (y +. clen) ;
	Cairo.rel_line_to cr two_clen (-1.0 *. two_clen) ;
	Cairo.stroke cr

let white_cross cr x y =
	Cairo.set_source_rgb cr 1.0 1.0 1.0 ;
	Cairo.set_line_width cr 3.0 ;
	do_cross cr x y 20.0

let black_cross cr x y =
	Cairo.set_source_rgb cr 0.0 0.0 0.0 ;
	Cairo.set_line_width cr 6.0 ;
	do_cross cr x y 22.0


let draw_position_marker gui cross =
	if gui.index < 0 || gui.index >= Array.length gui.crosses then () else
	let cr = Cairo_lablgtk.create gui.darea#misc#window in
	let p = gui.crosses.(gui.index) in
	Cairo.save cr ;
	if cross == White_cross then
		white_cross cr p.Complex.re p.Complex.im
	else
		black_cross cr p.Complex.re p.Complex.im ;
	Cairo.restore cr ;
	Cairo.show_page cr


let update_position_marker gui =
	if gui.index == Array.length gui.crosses - 1 then
	(	gui.complete <- true ;
		GMain.quit ()
		)
	else
	(	draw_position_marker gui Black_cross ;
		gui.index <- gui.index + 1 ;
		draw_position_marker gui White_cross
		)

let draw_text gui width height =
	let cr = Cairo_lablgtk.create gui.darea#misc#window in
	Cairo.set_source_rgb cr 1.0 1.0 1.0 ;
	Cairo.save cr ;
	(	Cairo.set_font_size cr 20.0 ;
		Cairo.move_to cr (0.5 *. float_of_int width) (0.5 *. float_of_int height -. 15.0) ;
		Cairo.show_text cr "Calibrating touchscreen" ;
		Cairo.move_to cr (0.5 *. float_of_int width) (0.5 *. float_of_int height +. 15.0) ;
		Cairo.show_text cr "Press <esc> to exit."
		) ;
	Cairo.restore cr

let key_press_cb ev =
	try
		let s = GdkEvent.Key.string ev in
		if int_of_char s.[0] == 27 then
		(	print_endline "Exiting as requested.\n" ;
			GMain.quit ()
			) ;
		true
	with Invalid_argument _ -> true

let button_press_cb gui ev =
	let x = GdkEvent.Button.x_root ev in
	let y = GdkEvent.Button.y_root ev in
	Printf.printf "button_press_cb : %4.0f %4.0f -> %4.0f %4.0f\n" gui.crosses.(gui.index).Complex.re gui.crosses.(gui.index).Complex.im x y ;
	gui.last_press <- { Complex.re = x ; im = y } ;
	true

let button_release_cb gui ev =
	let x = GdkEvent.Button.x_root ev in
	let y = GdkEvent.Button.y_root ev in
	if abs_float (gui.last_press.Complex.re -. x) < 25.0 && abs_float (gui.last_press.Complex.im -. y) < 25.0 then
	(	gui.clicks.(gui.index) <- { Complex.re = x ; im = y } ;
		update_position_marker gui
		) ;
	true

let gen_cross_array n width height =
	let max_index = float (n - 1) in
	let offset = 0.1 *. (max width height) in
	let right = width -. 2.0 *. offset in
	let bottom = height -. 2.0 *. offset in
	let entry i =
		let y = i / n in
		let x = i mod n in
		{ Complex.re = offset +. right *. float x /. max_index ; im = offset +. bottom *. float y /. max_index }
	in
	Array.init (n * n) entry

let redraw gui width height _ =
	let cr = Cairo_lablgtk.create gui.darea#misc#window in
	Cairo.rectangle cr 0.0 0.0 (float_of_int width) (float_of_int height) ;
	Cairo.set_source_rgb cr 0.0 0.0 0.0 ;
	Cairo.fill cr ;
	draw_text gui width height ;
	draw_position_marker gui White_cross ;
	true

let calc_correction orig_config calibrate_gain crosses clicks =
	let fp_valid f =
		match classify_float f with
		|	FP_normal -> true
		|	FP_zero -> true
		|	_ -> false
	in
	if Array.length crosses != 4 then
	(	print_endline "Error : Cannot generate correction data, expecting data for four points.\n" ;
		for k = 0 to Array.length clicks - 1 do
			let cross = crosses.(k) in
			let click = clicks.(k) in
			Printf.printf "  %4.0f, %4.0f  ->  %4.0f, %4.0f\n" cross.Complex.re cross.Complex.im click.Complex.re click.Complex.im ;
			done ;
		exit 1
		)
	else

	let new_config = Orientation.calc_config calibrate_gain orig_config crosses clicks in

	let good_config = fp_valid new_config.Config.x_gain
			&& fp_valid new_config.Config.x_offset
			&& fp_valid new_config.Config.y_gain
			&& fp_valid new_config.Config.y_offset
			&& abs_float new_config.Config.x_gain > 0.1
			&& abs_float new_config.Config.y_gain > 0.1
			&& abs_float new_config.Config.x_gain < 5.0
			&& abs_float new_config.Config.y_gain < 5.0
	in
	if good_config then
		(* We have a good result. *)
		true, new_config
	else
		false, orig_config


let ts_calibrate orig_config calibrate_gain =
	let width, height = get_screen_size () in

	let win = GWindow.window ~resizable:true () in
	ignore (win#connect#destroy GMain.quit) ;

	let area = GMisc.drawing_area ~width:width ~height:height ~packing:win#add () in
	let crosses = gen_cross_array 2 (float_of_int width) (float_of_int height) in
	let gui = {
		win = win ; darea = area ; crosses = crosses ;
		clicks = Array.make (Array.length crosses) { Complex.re = -1.0 ; im = -1.0 } ;
		last_press = { Complex.re = -1.0 ; im = -1.0 } ; index = 0 ; complete = false ;
		}
	in
	ignore (win#event#connect#key_press ~callback:key_press_cb) ;
	win#event#add [ `BUTTON_PRESS ; `BUTTON_RELEASE ] ;
	ignore (win#event#connect#button_press ~callback:(button_press_cb gui)) ;
	ignore (win#event#connect#button_release ~callback:(button_release_cb gui)) ;

	ignore (area#event#connect#expose (redraw gui width height)) ;

	win#show () ;

	let cursor = Gdk.Cursor.create `PLUS in
	Gdk.Window.set_cursor win#misc#window cursor ;

	(*
	**	Some window managers don't honour the fullscreen, but we should
	**	try anyway.
	*)
	win#fullscreen () ;

	(* If the fullscreen command didn't work this should. *)
	win#move 0 0 ;

	GMain.main () ;

	if gui.complete then
	(	let new_config =
		{	orig_config with
				 Config.top_lines = orig_config.Config.top_lines @ [ Printf.sprintf "# Screen size : %d x %d" width height ]
			}
		in
		let ok, data = calc_correction new_config calibrate_gain crosses gui.clicks in
		ok, data
		)
	else false, orig_config


let calibration_config old_config calibrate_gain =
	{	old_config with
			Config.swap_xy = false ;
			x_gain = calibrate_gain ; x_offset = 0.0 ;
			y_gain = calibrate_gain ; y_offset = 0.0 ;
		}


let () =
	if Unix.geteuid () != 0 then
	(	print_endline "\nSorry, need to run as root in order to run/stop\nthe daemon and write the config file.\n" ;
		exit 1
		)
	else
	let old_config = Config.read Version.config_name in
	let width, height = get_screen_size () in
	let calibrate_gain = (float_of_int (min width height)) /. (max touchscreen_height touchscreen_width) in
	Printf.printf "calibration gain : %f\n" calibrate_gain ;

	Config.write (calibration_config old_config calibrate_gain) Version.config_name ;
	let zytouchd = Util.get_process_id Version.daemon_name in
	List.iter Util.send_sig_hup zytouchd ;
	let ok, new_config = ts_calibrate old_config calibrate_gain in
	if not ok then
		Config.write old_config Version.config_name
	else
		Config.write new_config Version.config_name ;
	(* Tell daemon to re-read config. *)
	List.iter Util.send_sig_hup zytouchd


