(*
**	Copyright (C) 2006-2008  bCODE Pty Ltd (www.bcode.com)
**	Written and maintained by Erik de Castro Lopo <erikd@mega-nerd.com>
**
**	This program is free software: you can redistribute it and/or modify
**	it under the terms of the GNU General Public License as published by
**	the Free Software Foundation, either version 3 of the License, or
**	(at your option) any later version.
**
**	This program is distributed in the hope that it will be useful,
**	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
**	You should have received a copy of the GNU General Public License
**	along with this program.  If not, see <http://www.gnu.org/licenses/>.
*)

open ExtString
open ExtList


let ws_re = Str.regexp "[ \t\r\n]+"

let whitespace_nsplit str =
	Str.split ws_re str

let read_file_lines filename =
	let chan = open_in filename in
	let lst = ref [] in
	(	try
			while true do
				let raw_line = input_line chan in
				let line = String.strip raw_line in
				if String.length line > 0 then
					lst := line :: !lst ;
				done
		with End_of_file ->
			close_in chan
		) ;
	List.rev !lst

let read_command_output_lines cmd =
	let chan = Unix.open_process_in cmd in
	let lst = ref [] in
	(	try
			while true do
				let raw_line = input_line chan in
				let line = String.strip raw_line in
				if String.length line > 0 then
					lst := line :: !lst ;
				done
		with End_of_file ->
			ignore (Unix.close_process_in chan)
		) ;
	List.rev !lst

let with_out_filename filename writer =
	let f = open_out filename in
	try
		let result = writer f in
		close_out f ;
		result
	with
		any_exception ->
		close_out f ;
		raise any_exception

let write_whole_file data fname =
	let writer out_chan =
		output_string out_chan data
	in
	with_out_filename fname writer

let file_path_split path =
	try
		let slen = String.length path in
		let last_slash = String.rindex path '/' in
		String.sub path 0 last_slash, String.sub path (last_slash + 1) (slen - last_slash - 1)
	with Not_found ->
		"", path

let get_process_id process_name =
	let find_pids line =
		let entries = whitespace_nsplit line in
		int_of_string (List.nth entries 1)
	in
	let cmd = Printf.sprintf "ps auwx | grep %s | grep -v grep" process_name in
	let lines = read_command_output_lines cmd in
	List.map find_pids lines

let send_sig_hup process_id =
	try
		Unix.kill process_id Sys.sighup
	with
		Unix.Unix_error (e, n, a) ->
			Printf.printf "Unix error (%s -HUP %d) : %s\n" n process_id (Unix.error_message e)

let dir_exists dirname =
	try
		let stat = Unix.stat dirname in
		if stat.Unix.st_kind == Unix.S_DIR then true
		else false
	with _ ->
		false

let rec_mkdir dirname =
	let rec mksub head lst =
		match lst with
		|	[] -> ()
		|	hd :: tl ->
				let new_head = head ^ "/" ^ hd in
				if not (dir_exists new_head) then
					Unix.mkdir new_head 0o755 ;
				mksub new_head tl
	in
	if String.length dirname > 0 then
	(	let pathlist = List.filter (fun s -> String.length s > 0) (String.nsplit dirname "/") in
		let root = if dirname.[0] == '/' then "" else "." in
		mksub root pathlist
		)
