/*
**	Copyright (C) 2006-2008  bCODE Pty Ltd (www.bcode.com)
**	Written and maintained by Erik de Castro Lopo <erikd@mega-nerd.com>
**
**	This program is free software: you can redistribute it and/or modify
**	it under the terms of the GNU General Public License as published by
**	the Free Software Foundation, either version 3 of the License, or
**	(at your option) any later version.
**
**	This program is distributed in the hope that it will be useful,
**	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
**	You should have received a copy of the GNU General Public License
**	along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <math.h>

#include <unistd.h>
#include <signal.h>

#include <errno.h>
#include <assert.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <usb.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XTest.h>

#include "zytouch_usb.h"
#include "debug.h"

#define ARRAY_LEN(x)   ((int) (sizeof (x) / sizeof (x [0])))

#define	X_AXIS_BINS			16
#define	Y_AXIS_BINS			16

#define	WRITE_ENDPOINT	1
#define	READ_ENDPOINT	0x81

#define	USB_BULK_WAIT_MS	200

/* Global variable. Set in SIGHUP handler, reset when config has been loaded. */
int g_reload_config = 0 ;

/*
**	Each call to usb_bulk_read() returns 8 bytes of data. The first two bytes
**	are header bytes and the last 6 are the real data bytes.
*/
typedef struct
{	unsigned char header [2] ;
	unsigned char data [6] ;
} BULKBUF ;

typedef struct
{	int button_down ;
	float x, y ;
} POINT ;

typedef struct
{
	/*
	**	Call (0,0) top left hand corner of screen.
	**
	** Data from touchsceen comes back with 16 y-axis bins and 16 x-axis
	** bins.
	**
	**                                    x-inf                         x0
	**	 "y-inf                         y0                                "
	**    000102030405060708090a0b0c0d0e0f
	**                                    000102030405060708090a0b0c0d0e0f
	*/
	union
	{	unsigned char data [X_AXIS_BINS + Y_AXIS_BINS] ;
		struct
		{	/*
			**	This ordering is important.
			**	Its specified by what we get back from the hardware.
			*/
			unsigned char ydata [Y_AXIS_BINS] ;
			unsigned char xdata [X_AXIS_BINS] ;
		} ;
	} ;
} FULLSCAN ;

typedef struct
{
	int mem_current ;
	float xmem [X_AXIS_BINS][4] ;
	float ymem [Y_AXIS_BINS][4] ;

	float xdata [X_AXIS_BINS] ;
	float ydata [Y_AXIS_BINS] ;
} SCANFILTER ;

typedef struct
{	int count ;
	int clicked ;
	float xout, yout ;
	float certainty ;
} FILTER ;

static int caught_sighup = 0 ;


static void
sighup_handler (int unused __attribute ((unused)))
{	dprintf ("Caught SIGHUP\n") ;
	caught_sighup = 1 ;
} /* sighup_handler */


static inline int
limit_int (int x, int lower, int upper)
{
	if (x < lower) return lower ;
	if (x > upper) return upper ;
	return x ;
}

static inline char
ascii_nibble (unsigned char nib)
{
	nib = nib & 0xf ;
	if (nib > 9)
		return nib - 10 + 'a' ;
	return nib + '0' ;
}

static void
string_of_uchar (const unsigned char *bin, int len, char * str, int with_space, int sense_threshold)
{
	int k ;
	unsigned char ch ;

	for (k = 0 ; k < len ; k++)
	{	ch = bin [k] ;

		if (with_space && ch < sense_threshold)
		{	str [2 * k] = ' ' ;
			str [2 * k + 1] = ' ' ;
			}
		else
		{	str [2 * k] = ascii_nibble (ch >> 4) ;
			str [2 * k + 1] = ascii_nibble (ch) ;
			} ;
		} ;

	str [2 * len] = 0 ;
} /* string_of_uchar */

static int min_int (int a, int b)
{	return a < b ? a : b ;
}

static float min_float (float a, float b)
{	return a < b ? a : b ;
}

static int max_int (int a, int b)
{	return a > b ? a : b ;
}

#define	USB_READ_LINES 		6
#define	USB_DATA_READLEN 	8

#define MIN_RAMP_RATE		25

static void
gain_adjust (FULLSCAN *udiff, const CONFIG_V2 * config)
{
	int k ;

	/* Gain adjustment really doesn't seem very useful. */

	/* Right hand side X gain adjustment. */
	udiff->xdata [0] = lrintf (config->xedgegain / 10 * udiff->xdata [0]) ;
	k = sizeof (udiff->xdata) - 1 ;

	/* Left hand side X gain adjustment. */
	udiff->xdata [k] = lrintf (config->xedgegain / 10 * udiff->xdata [k]) ;

	/* Right hand side Y gain adjustment. */
	udiff->ydata [0] = lrintf (config->yedgegain / 10 * udiff->ydata [0]) ;
	k = sizeof (udiff->ydata) - 1 ;

	/* Left hand side Y gain adjustment. */
	udiff->ydata [k] = lrintf (config->yedgegain / 10 * udiff->ydata [k]) ;

} /* gain_adjust */

static int
apply_filter (FILTER *filt, float xin, float yin)
{
	float coeff, xdiff, ydiff ;

	/*
	**	The filter counts the inputs and doesn't actually produce a valid
	**	output until it has seen 4 valid inputs.
	**	This is reset elsewhere by setting filt->count to zero.
	*/

	if (filt->count <= 0)
	{	filt->count ++ ;
		filt->xout = xin ;
		filt->yout = yin ;
		return 0 ;
		} ;

	xdiff = (filt->xout - xin) / 800.0 ;
	ydiff = (filt->yout - yin) / 640.0 ;
	filt->certainty = pow (1.01 / (1.0 + xdiff * xdiff + ydiff * ydiff), 4.0) ;

	/*
	**	Basically a first lowpass filter with an adaptive coefficent.
	**
	**		out [n] = coeff * in [n] + (1.0 - coeff) * out [n - 1]
	**
	**	The coeffient is adapted to be large when the difference between
	**	in [n] and out [n-1] is large and smaller when the difference
	**	is small.
	*/

	coeff = 0.5 ;

	if (coeff > 0.95)
		coeff = 0.95 ;

	filt->xout = coeff * xin + (1.0 - coeff) * filt->xout ;
	filt->yout = coeff * yin + (1.0 - coeff) * filt->yout ;

	/* If we have less than 2 inputs return invalid output. */
	if (filt->count < 1)
	{	filt->count ++ ;
		return 0 ;
		} ;

	/* Return a valid output. */
	return 1 ;
} /* apply_filter */


static void
filter_scan_data (SCANFILTER * fdiff, const FULLSCAN *udiff)
{
	int k ;

	/* actual filter code removed and replaced with simple copy of udiff into fdiff - SE */

	for (k = 0 ; k < ARRAY_LEN (udiff->xdata) ; k++)
	{
		fdiff->xdata [k] = udiff->xdata [k];
	}

	for (k = 0 ; k < ARRAY_LEN (udiff->ydata) ; k++)
	{
		fdiff->ydata [k] = udiff->ydata [k];
	}

}


static FILTER filter = { 0, 0, 0.0, 0.0, 0.0 } ;
static POINT point = { 0, 0.0, 0.0 } ;

static SCANFILTER fdiff ;

static void
move_mouse (Display *display, const FULLSCAN *udiff, const CONFIG_V2 * config)
{
	unsigned  k ;
	float xmax, xmin, ymax, ymin ;
	int xmaxpos, ymaxpos ;
	int scaled_xpos, scaled_ypos ;
	float fxmaxpos, fymaxpos ;
	float diffsum, weightedsum ;	// working variables for centre of mass calc


	filter_scan_data (&fdiff, udiff) ;

	ymaxpos = 0 ;
	ymax = ymin = fdiff.ydata [0] ;
	diffsum = weightedsum = 0 ;
	for (k = 1 ; k < ARRAY_LEN (fdiff.ydata) ; k++)
	{	if (ymax < fdiff.ydata [k])
		{	ymaxpos = k ;
			ymax = fdiff.ydata [k] ;
			} ;
		ymin = min_float (ymin, fdiff.ydata [k]) ;
		} ;

	// handle case if two equal values
	if (ymaxpos > 0 && ymaxpos < (ARRAY_LEN (fdiff.ydata)) - 1)
	{	if (fdiff.ydata [ymaxpos] == fdiff.ydata [ymaxpos + 1] && fdiff.ydata [ymaxpos + 2] > fdiff.ydata [ymaxpos - 1])
			ymaxpos++;
		} ;

	// use centre of mass of 5 values above threshold around maxpos
	for (k = 0 ; k < 5 ; k++)
	{	if ( (ymaxpos + k - 2 > 0) && (ymaxpos + k - 2 < ARRAY_LEN (fdiff.ydata)) )
		{	if (fdiff.ydata [ymaxpos + k - 2] >= config->sensitivity)
			{	diffsum += (fdiff.ydata [ymaxpos + k - 2] - config->sensitivity) ;
				weightedsum += (fdiff.ydata [ymaxpos + k - 2] - config->sensitivity) * (ymaxpos + k - 2) ;
				} ;
			} ;
		};

	if (diffsum > 0)
		fymaxpos = weightedsum / diffsum ;
	else
	{	filter.count = 0 ;
		if (point.button_down == 1)
		{	XTestFakeButtonEvent (display, 1, False, CurrentTime) ;
			dprintf ("Button release") ;
			XFlush (display) ;
			} ;
		dprintf ("\n") ;
		point.button_down = 0 ;
		return ;
		} ;

	xmaxpos = 0 ;
	xmax = xmin = fdiff.xdata [0] ;
	diffsum = weightedsum = 0 ;
	for (k = 1 ; k < ARRAY_LEN (fdiff.xdata) ; k++)
	{	if (xmax < fdiff.xdata [k])
		{	xmaxpos = k ;
			xmax = fdiff.xdata [k] ;
			} ;
		xmin = min_float (xmin, fdiff.xdata [k]) ;
		} ;
	// handle case if two equal values
	if (xmaxpos > 0 && xmaxpos < (ARRAY_LEN (fdiff.xdata)) - 1)
	{	if (fdiff.xdata [xmaxpos] == fdiff.xdata [xmaxpos + 1] && fdiff.xdata [xmaxpos + 2] > fdiff.xdata [xmaxpos - 1])
			xmaxpos++;
	} ;

	// use centre of mass of 5 values above threshold around maxpos
	for (k = 0 ; k < 5 ; k++)
	{	if ( (xmaxpos + k - 2 > 0) && (xmaxpos + k - 2 < ARRAY_LEN (fdiff.xdata)) )
		{	if (fdiff.xdata [xmaxpos + k - 2] >= config->sensitivity)
			{	diffsum += (fdiff.xdata [xmaxpos + k - 2] - config->sensitivity) ;
				weightedsum += (fdiff.xdata [xmaxpos + k - 2] - config->sensitivity) * (xmaxpos + k - 2) ;
				} ;
			} ;
		};

	if (diffsum > 0)
		fxmaxpos = weightedsum / diffsum ;
	else
	{	filter.count = 0 ;
		if (point.button_down == 1)
		{	XTestFakeButtonEvent (display, 1, False, CurrentTime) ;
			dprintf ("Button release") ;
			XFlush (display) ;
			} ;
		dprintf ("\n") ;
		point.button_down = 0 ;
		return ;
		} ;

	dprintf ("%4d, %4d     ->    ", xmaxpos, ymaxpos) ;

	/* If nothing above the sense threshold, then exit here */
	if (ymax <= config->range || xmax <= config->range)
	{	filter.count = 0 ;
		if (point.button_down == 1)
		{	XTestFakeButtonEvent (display, 1, False, CurrentTime) ;
			dprintf ("Button release") ;
			XFlush (display) ;
			} ;
		dprintf ("\n") ;
		point.button_down = 0 ;
		return ;
		} ;

	dprintf ("%6.2f", fxmaxpos) ;
	/* Flip left/right (or top/bottom) and scale. */
	fxmaxpos = (ARRAY_LEN (fdiff.xdata) - fxmaxpos) * config->xscale / 10 + config->xoff ;
	if (fxmaxpos < 0.0)
		fxmaxpos = 0.0 ;

	dprintf (", ") ;

	dprintf ("%6.2f", fymaxpos) ;
	/* Flip left/right (or top/bottom) and scale. */
	fymaxpos = (ARRAY_LEN (fdiff.ydata) - fymaxpos) * config->yscale / 10 + config->yoff ;
	if (fymaxpos < 0.0)
		fymaxpos = 0.0 ;

	if (apply_filter (&filter, fxmaxpos, fymaxpos) == 0)
	{	dprintf ("\n") ;
		return ;
		} ;

	scaled_xpos = lrintf (filter.xout) ;
	scaled_ypos = lrintf (filter.yout) ;

	if (scaled_xpos < 0 || scaled_ypos < 0)
		dprintf ("   %f, %f -> %d %d *****\n", filter.xout, filter.yout, scaled_xpos, scaled_ypos) ;
	else
	{
		/* Send the events to X. */
		dprintf ("    ->    %4d, %4d    (%4.2f)\n", scaled_xpos, scaled_ypos, filter.certainty) ;
		XTestFakeMotionEvent (display, -1, scaled_xpos, scaled_ypos, 0) ;

		if (point.button_down == 0)
		{	point.button_down = 1 ;
			point.x = scaled_xpos ;
			point.y = scaled_ypos ;
			XTestFakeButtonEvent (display, 1, True, CurrentTime) ;
			} ;

		XFlush (display) ;
		} ;

} /* move_mouse */


static FULLSCAN min_data ;
static FULLSCAN max_data ;
static FULLSCAN correction ;
static FULLSCAN max_udiff ;
static int min_ramp_count;

static void
init_min_max_correction (void)
{
	memset (&min_data, 0xff, sizeof (min_data)) ;
	memset (&max_data, 0, sizeof (max_data)) ;
	memset (&correction, 0, sizeof (correction)) ;
	memset (&max_udiff, 0, sizeof (max_udiff)) ;
	min_ramp_count = 0;
} /* init_min_max_correction */

static void
print_output (Display * display, const FULLSCAN * fullscan, const CONFIG_V2 * config)
{
	FULLSCAN udiff, current ;
	static char str [2 * sizeof (udiff) + 1] ;
	int max_diff ;
	unsigned k ;

	max_diff = 0 ;

	for (k = 0 ; k < sizeof (fullscan->data) ; k ++)
	{
		current.data [k] = (fullscan->data [k] + correction.data [k]) & 0xff ;

		min_data.data [k] = min_int (min_data.data [k], current.data [k]) ;
		max_data.data [k] = max_int (max_data.data [k], current.data [k]) ;

		/*
		**	Sometimes the min and the max can be very near zero and a new fullscan comes
		**	in with value near 0xff.
		**	When this happens, swap tha max and the min.
		*/
		if (max_data.data [k] - min_data.data [k] > 0xA0)
		{
			unsigned char temp = min_data.data [k] ;
			min_data.data [k] = max_data.data [k] ;
			max_data.data [k] = temp ;
			} ;

		if (min_data.data [k] > 0x90)
		{	correction.data [k] += 0x80 ;
			min_data.data [k] -= 0x80 ; ;
			max_data.data [k] -= 0x80 ;
			} ;

		udiff.data [k] = max_data.data [k] - min_data.data [k] ;

		max_udiff.data [k] = max_int (max_udiff.data [k], udiff.data [k]) ;

		max_diff = max_int (udiff.data [k], max_diff) ;
		} ;

	if (1)
	{
		string_of_uchar (current.data, sizeof (fullscan->data), str, 0, config->sensitivity) ;
		dprintf ("\n current '%s'\n", str) ;

#if 0
		string_of_uchar (min_data.data, sizeof (fullscan->data), str, 0, config->sensitivity) ;
		dprintf ("     min '%s'\n", str) ;

		string_of_uchar (max_data.data, sizeof (fullscan->data), str, 0, config->sensitivity) ;
		dprintf ("     max '%s'\n", str) ;
#endif

		string_of_uchar (udiff.data, sizeof (udiff.data), str, 1, config->sensitivity) ;
		dprintf ("   udiff '%s'\n", str) ;

		string_of_uchar (max_udiff.data, sizeof (udiff.data), str, 1, config->sensitivity) ;
		dprintf ("max diff '%s'\n", str) ;
		} ;

	gain_adjust (&udiff, config) ;
	move_mouse (display, &udiff, config) ;

	/* Decay the min and the max towards each other. */

	for (k = 0 ; k < sizeof (min_data) ; k ++)
	{	/*
		**	Ramps the min up one count every MIN_RAMP_COUNT samples to account
		**	for drift over time.
		*/
		if (min_ramp_count > MIN_RAMP_RATE)
			min_data.data [k] ++ ;

		/* Decays max immediately - really means that max should always reflect current */
		max_data.data [k] = min_data.data [k] ;
		} ;

	if (min_ramp_count > MIN_RAMP_RATE)
		min_ramp_count = 0 ;
	else
		min_ramp_count += 1;

	return ;
} /* print_output */

static void
wait_for_reset (struct usb_dev_handle *handle)
{	static char reset_data [USB_DATA_READLEN] = { 0x80, 0x01, 0x08, 0x00, 0x20, 0x00, 0x00, 0x00 } ;
	BULKBUF bulkbuf ;
	int k, retval ;

	assert (sizeof (bulkbuf) == 8) ;

	if ((retval = usb_resetep (handle, READ_ENDPOINT)) < 0)
		dprintf ("usb_resetep : %d %s\n", retval, strerror (errno)) ;

	/* This seems to be necessary. */
	for (k = 0 ; k < 100 ; k++)
	{	retval = usb_bulk_read (handle, READ_ENDPOINT, (char *) &bulkbuf, sizeof (bulkbuf), USB_BULK_WAIT_MS) ;

		if (retval == sizeof (bulkbuf) && bulkbuf.header [0] == 0x80)
			break ;
		} ;

	if ((retval = usb_bulk_write (handle, WRITE_ENDPOINT, reset_data, sizeof (reset_data), USB_BULK_WAIT_MS)) < 0)
		dprintf ("usb_bulk_write returned %d : %s\n", retval, strerror (errno)) ;

	/* After reset, dispose of first bunch of reads. */
	for (k = 0 ; k < 50 ; k++)
	{	retval = usb_bulk_read (handle, READ_ENDPOINT, (char *) &bulkbuf, sizeof (bulkbuf), USB_BULK_WAIT_MS) ;

		if (retval == sizeof (bulkbuf) && bulkbuf.header [0] == 0x80)
			break ;
		} ;

} /* wait_for_reset */

static void
ver2_run (struct usb_dev_handle *handle, Display * display, const CONFIG_V2 * config)
{	FULLSCAN fullscan ;
	BULKBUF bulkbuf ;
	int retval ;
	int sequence = 0 ;

	memset (&fullscan, 0, sizeof (fullscan)) ;

	assert (sizeof (fullscan) == sizeof (fullscan.xdata) + sizeof (fullscan.ydata)) ;

	if ((retval = usb_claim_interface (handle, 0)) != 0)
	{	dprintf ("usb_claim_interface (handle, 0) return %d (%s).\n", retval, strerror (errno), 0) ;
		return ;
		} ;

	dprintf ("Setting endpoint address to %d because thats what lsusb tells me.\n", READ_ENDPOINT) ;

	wait_for_reset (handle) ;

	while (1)
	{	usleep (1000) ;
		retval = usb_bulk_read (handle, READ_ENDPOINT, (char*) &bulkbuf, sizeof (bulkbuf), USB_BULK_WAIT_MS) ;

		if (retval == -ENODEV)
		{	dprintf ("usb_bulk_read : ENODEV\n") ;
			break ;
			} ;

		if (retval != sizeof (bulkbuf))
			continue ;

		if (sequence == 0 && bulkbuf.header [0] == 0)
		{	/* If sequence == 0 && bulkbuf.header [0] == 0 then we're reading too fast. */
			sequence = 0 ;
			usleep (1000) ;
			}
		else if (bulkbuf.header [0] == 0 || bulkbuf.header [0] == 0x80)
			sequence = 0 ;
		else if (bulkbuf.header [0] == 0xd6)
		{	memcpy (fullscan.data, bulkbuf.data, sizeof (bulkbuf.data)) ;
			sequence = 1 ;
			}
		else
		{	switch ((sequence << 16) + bulkbuf.header [0])
			{	case ((1 << 16) + 0xc6) :
				case ((2 << 16) + 0xc6) :
				case ((3 << 16) + 0xc6) :
				case ((4 << 16) + 0xc6) :
					memcpy (fullscan.data + sequence * sizeof (bulkbuf.data), bulkbuf.data, sizeof (bulkbuf.data)) ;
					sequence ++ ;
					break ;

				case ((5 << 16) + 0xca) :
					/* Weird ass special case. We sometimes get 'ca1ebe8c20200000' and want to ignore it. */
					if (bulkbuf.data [5] == 0)
						break ;

					/* Otherwise ..... */
					memcpy (fullscan.data + sequence * sizeof (bulkbuf.data), bulkbuf.data, sizeof (bulkbuf.data) - 2) ;
					print_output (display, &fullscan, config) ;
					sequence = 0 ;
					break ;

				/* Sequence recovery cases. */
				case ((0 << 16)	+ 0xc6):
				case ((0 << 16)	+ 0xca):
				case ((1 << 16) + 0xca) :
				case ((2 << 16) + 0xca) :
				case ((3 << 16) + 0xca) :
				case ((4 << 16) + 0xca) :
					sequence = 0 ;
					break ;

				default :
					dprintf ("sequence : %d    bulkbuf.header [0] : %02x\n", sequence, bulkbuf.header [0]) ;
					break ;
				} ;
			} ;
		} ;

	usb_release_interface (handle, 0) ;

	return ;
} /* ver2_run */


static void
ver3_xtest (Display * display, const POINT * point)
{	static int last_button_down = 0 ;
	static int down_x = 0, down_y = 0;
	int scaled_xpos, scaled_ypos ;

	if (last_button_down == 0 && point->button_down == 1)
	{	down_x = point->x ;
		down_y = point->y ;
		/*-dprintf ("Button press   : %3d, %3d\n", x, y) ;-*/
		}
	else if (last_button_down == 1 && point->button_down == 0 && abs (down_x - point->x) < 30 && abs (down_y - point->y) < 30)
	{	XTestFakeButtonEvent (display, 1, True, CurrentTime) ;
		XTestFakeButtonEvent (display, 1, False, CurrentTime) ;
		/*-dprintf ("Button release : %3d, %3d\n", x, y) ;-*/
		} ;

	scaled_xpos = lrintf (point->x) ;
	scaled_ypos = lrintf (point->y) ;

	/* XTestFakeMotionEvent (display, screen_number, x, y, delay) ; */
	XTestFakeMotionEvent (display, -1, scaled_xpos, scaled_ypos, 0) ;

	XFlush (display) ;
	last_button_down = point->button_down ;

	return ;
} /* ver3_xtest */


static int
ver3_read_button_x_y (struct usb_dev_handle *handle, POINT * point, const CONFIG_V3 * config)
{
	/* lsusb -d 14c8:0003 -vvvv tells us that endpoint 0x81 is 64 bytes, but position info is only 5 bytes. */
	unsigned char buffer [64] ;
	int retval, zero_retval_count = 0 ;
	int read_x, read_y ;

	buffer [0] = 0 ;

	for ( ; ; )
	{	retval = usb_interrupt_read (handle, READ_ENDPOINT, (char *) buffer, sizeof (buffer), 2000) ;

		if (caught_sighup)
			return 0 ;

		if (retval == 0)
		{	zero_retval_count ++ ;
			if (zero_retval_count < 5)
				continue ;

			dprintf ("%s : usb_interrupt_read returned zero %d times.\n", __func__, zero_retval_count) ;

			/* Return I/O error to force the wrapping code to reacquire the usb device. */
			return -EIO ;
			} ;

		if (retval == -ETIMEDOUT)
			/* Ignore timeouts. */
			continue ;

		if (retval < 0)
		{	dprintf ("usb_interrupt_read : %d (%s)\n", retval, strerror (errno)) ;
			return retval ;
			} ;

		zero_retval_count = 0 ;

		if ((buffer [0] & 0x80) != 0x80)
		{	/* Basically benign. */
			retval = retval >= ARRAY_LEN (buffer) ? ARRAY_LEN (buffer) - 1 : retval ;

			do
			{	buffer [retval] = 0 ;	
				retval -- ;
			} while (isspace (buffer [retval])) ;

			/* Ignore heartbeat from touchscreen. */
			if (buffer [0] == ':' && buffer [1] == 'I')
				continue ;

			dprintf ("%s : msg '%s'\n", __func__, buffer) ;
			continue ;
			} ;

		break ;
		} ;

	point->button_down = (buffer [0] & 0x40) ? 1 : 0 ;

	read_x = (buffer [3] & 0x7f) + ((buffer [4] & 0x7) << 7) ;
	read_y = (buffer [1] & 0x7f) + ((buffer [2] & 0x7) << 7) ;
	
	if (config->swap_xy == 0)
	{	point->x = config->x_offset + config->x_gain * read_x ;
		point->y = config->y_offset + config->y_gain * read_y ;
		}
	else
	{	point->y = config->y_offset + config->y_gain * read_x ;
		point->x = config->x_offset + config->x_gain * read_y ;
		} ;

	dprintf ("b : %1d    read : %4d, %4d    ->    X : %4.0f, %4.0f\n", point->button_down, read_x, read_y, point->x, point->y) ;

	return 0 ;
} /* ver3_read_button_x_y */

static usb_dev_handle*
find_handle (ZYTOUCH * zyt)
{	struct usb_bus * pbus ;
	struct usb_device *pdev = NULL ;

	usb_init () ;
	usb_find_busses () ;
	usb_find_devices () ;
	pbus = usb_get_busses () ;

	if (pbus == NULL)
	{	dprintf ("%s %d : pbus == NULL.\n", __func__, __LINE__) ;
		return NULL ;
		} ;

	dprintf ("Have bus pointer.\n") ;

	/* Find the device we're interested in. */
	for ( ; pbus != NULL ; pbus = pbus->next)
		for (pdev = pbus->devices ; pdev != NULL ; pdev = pdev->next)
		{
			usb_dev_handle *handle = NULL ;
			char buffer [256] ;
			int retval ;

			memset (buffer, 0, sizeof (buffer)) ;

			handle = usb_open (pdev) ;
			if (handle == NULL)
				continue ;

			if (pdev->descriptor.idVendor == 0x14c8 && pdev->descriptor.idProduct == 0x0002)
			{	dprintf ("Have device handle for : '%04x:%04x'.\n", pdev->descriptor.idVendor, pdev->descriptor.idProduct) ;
				zyt->vendor = pdev->descriptor.idVendor ;
				zyt->product = pdev->descriptor.idProduct ;
				return handle ;
				} ;

			if (pdev->descriptor.idVendor == 0x14c8 && pdev->descriptor.idProduct == 0x0003)
			{	dprintf ("Have device handle for : '%04x:%04x'.\n", pdev->descriptor.idVendor, pdev->descriptor.idProduct) ;
				zyt->vendor = pdev->descriptor.idVendor ;
				zyt->product = pdev->descriptor.idProduct ;
				return handle ;
				} ;

			retval = usb_get_string_simple (handle, 1, buffer, sizeof (buffer)) ;
			if (retval > 0)
			{
				buffer [retval] = 0 ;

				if (strcmp (buffer, "Binstead") == 0)
				{	dprintf ("Have device handle (%04x:%04x).\n", pdev->descriptor.idVendor, pdev->descriptor.idProduct) ;
					return handle ;
					} ;
				} ;

			usb_close (handle) ;
			} ;

	dprintf ("Binstead not found.\n") ;

	return NULL ;
} /* find_handle */


int
find_handle_retry (ZYTOUCH * zyt)
{
	if (zyt == NULL)
	{	printf ("%s : zyt == NULL.\n", __func__) ;
		return 1 ;
		} ;

	memset (zyt, 0, sizeof (*zyt)) ;

	while (zyt->handle == NULL)
	{
		zyt->handle = find_handle (zyt) ;
		if (zyt->handle == NULL)
		{	dprintf ("%s : Initial NULL, sleeping for 5 seconds.\n", __func__) ;
			sleep (5) ;
			continue ;
			} ;

		dprintf ("%s : Doing device reset.\n", __func__) ;

		/* Reset device. */
		usb_reset (zyt->handle) ;
		sleep (1) ;

		/* Need to regain handle after reset. */
		zyt->handle = find_handle (zyt) ;
		if (zyt->handle == NULL)
		{	dprintf ("%s : Second NULL, sleeping for 5 seconds.\n", __func__) ;
			sleep (5) ;
			} ;
		} ;

	return 0 ;
} /* find_handle_retry */



static void
ver2_do_device (struct usb_dev_handle *handle, Display *display, const CONFIG_COMMON * common)
{
	CONFIG_V2 config_v2 ;

	read_v2_config ("/etc/zytouchd.conf", &config_v2) ;
	config_v2.common = *common ;

	dprintf ("sense threshold : %d\n", config_v2.sensitivity) ;
	dprintf ("sense range : %d\n", config_v2.range) ;
	dprintf ("x edge gain * 10 : %d\n", config_v2.xedgegain) ;
	dprintf ("y edge gain * 10 : %d\n", config_v2.yedgegain) ;
	dprintf ("xscale * 10 : %d\n", config_v2.xscale) ;
	dprintf ("xoffset : %d\n", config_v2.xoff) ;
	dprintf ("yscale * 10 : %d\n", config_v2.yscale) ;
	dprintf ("yoffset : %d\n", config_v2.yoff) ;

	init_min_max_correction () ;
	ver2_run (handle, display, &config_v2) ;
} /* ver2_do_device */

static int
ver3_dump_settings (struct usb_dev_handle *handle)
{	char buffer [64] ;
	int retval ;

	/* Request non-volatile settings. */
	retval = usb_control_msg (handle, 0x40, 0, 0x58, 0, buffer, 0, 1000) ;
	if (retval < 0)
	{	dprintf ("%s %d : usb_control_msg returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;
		return retval ;
		} ;

	memset (buffer, 0, sizeof (buffer)) ;

	while ((retval = usb_interrupt_read (handle, READ_ENDPOINT, (char *) buffer, sizeof (buffer), 10000)) <= 0)
		/* Do nothing */ ;

	if (retval > 0)
	{	buffer [retval >= ARRAY_LEN (buffer) ? ARRAY_LEN (buffer) - 1 : retval] = 0 ;

		dprintf ("Sensor wires     : %d\n", int_of_string (buffer + 2, 5)) ;
		dprintf ("Glass thickness  : %d\n", int_of_string (buffer + 7, 5)) ;
		dprintf ("Threshold        : %d\n", int_of_string (buffer + 12, 5)) ;
		dprintf ("Frame average    : %d\n", int_of_string (buffer + 17, 5)) ;
		dprintf ("Enabled          : %d\n", int_of_string (buffer + 22, 5)) ;
		dprintf ("Stabilization    : %d\n", int_of_string (buffer + 32, 5)) ;
		dprintf ("Ack/Nak          : %d\n", int_of_string (buffer + 37, 5)) ;
		} ;

	return 0 ;
} /* ver3_dump_settings */

static int
ver3_setup (struct usb_dev_handle *handle, const CONFIG_V3 * config)
{	char buffer [64] ;
	int retval, k ;
	int firmware, tweaked = 0 ;

	memset (buffer, 0, sizeof (buffer)) ;

	/* Grab firmware version etc. */
	retval = usb_control_msg (handle, 0x40, 0, 0x56, 0, buffer, 0, 1000) ;
	if (retval < 0)
	{	dprintf ("%s %d : usb_control_msg returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;
		return retval ;
		} ;

	memset (buffer, 0, sizeof (buffer)) ;

	retval = usb_interrupt_read (handle, READ_ENDPOINT, (char *) buffer, sizeof (buffer), 1000) ;
	if (retval < 0)
	{	dprintf ("%s %d : usb_interrupt_read returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;
		return retval ;
		} ;

	buffer [retval >= ARRAY_LEN (buffer) ? ARRAY_LEN (buffer) - 1 : retval] = 0 ;
	dprintf ("Firmware version : %d.%02d\n", int_of_string (buffer + 2, 3), int_of_string (buffer + 5, 2)) ;
	dprintf ("Serial number    : %010d\n", int_of_string (buffer + 7, 10)) ;
	dprintf ("Controller type  : %d\n\n", int_of_string (buffer + 17, 5)) ;

	firmware = (int_of_string (buffer + 2, 3) << 16) + int_of_string (buffer + 5, 2) ;

	for (k = 0 ; k < 5 ; k++)
	{
		retval = usb_interrupt_read (handle, READ_ENDPOINT, (char *) buffer, sizeof (buffer), 200) ;
		if (retval >= 0)
			break ;
		else if (retval < 0 && retval != -ETIMEDOUT)
		{	dprintf ("%s %d : usb_interrupt_read returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;
			return retval ;
			} ;
	} ;

#if 0
	dprintf ("Restoring factory defaults.\n") ;
	/* Restore factory defaults. */
	retval = usb_control_msg (handle, 0x40, 0, 0x44, 0, buffer, 0, 1000) ;
	if (retval < 0)
	{	dprintf ("%s %d : usb_control_msg returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;
		return retval ;
		} ;
#endif

	if (0)
		ver3_dump_settings (handle) ;

	if (firmware >= 0x10003)
	{
		tweaked = 0 ;

		/* Glass thickness : thick. */
		/*-dprintf ("\nglass thikness : %d -> %d\n", config->glass_thickness, 230 + limit_int (config->glass_thickness, 0, 3)) ;-*/
		retval = usb_control_msg (handle, 0x40, 0, 230 + limit_int (config->glass_thickness, 0, 3), 0, "", 0, 1000) ;
		if (retval < 0)
			dprintf ("%s %d : usb_control_msg returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;

		/* Touch threshold : value - 100. */
		/*-dprintf ("threshold : %d -> %d\n", config->threshold, 100 + limit_int (config->threshold, 0, 50)) ;-*/
		retval = usb_control_msg (handle, 0x40, 0, 100 + limit_int (config->threshold_fw3, 0, 50), 0, "", 0, 1000) ;
		if (retval < 0)
			dprintf ("%s %d : usb_control_msg returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;

		/* Frame averaging : value - 220. */
		/*-dprintf ("frame_avg : %d -> %d\n", config->frame_avg, 220 + limit_int (config->frame_avg, 0, 29)) ;-*/
		retval = usb_control_msg (handle, 0x40, 0, 220 + limit_int (config->frame_avg, 0, 29), 0, "", 0, 1000) ;
		if (retval < 0)
			dprintf ("%s %d : usb_control_msg returns %d (%s).\n", __func__, __LINE__, retval, strerror (errno)) ;
		} ;

	if (tweaked)
		dprintf ("\nAfter tweaking values.\n\n") ;

	ver3_dump_settings (handle) ;

	return 0 ;
} /* ver3_setup */


static void
ver3_do_device (struct usb_dev_handle *handle, Display * display, const CONFIG_COMMON * common)
{
	CONFIG_V3 config ;
	POINT point = { 0, 0.0, 0.0 } ;
	int retval ;

	read_v3_config ("/etc/zytouchd_v3.conf", &config) ;
	config.common = *common ;

	if ((retval = usb_claim_interface (handle, 0)) != 0)
	{	dprintf ("usb_claim_interface (handle, 0) return %d (%s).\n", retval, strerror (errno)) ;
		return ;
		} ;

	retval = ver3_setup (handle, &config) ;

	while (retval == 0)
	{	retval = ver3_read_button_x_y (handle, &point, &config) ;

		if (caught_sighup == 1)
		{	dprintf ("%s : Caught SIGHUP.\n", __func__) ;
			break ;
			} ;

		if (retval == 0)
			ver3_xtest (display, &point) ;
		} ;

	usb_release_interface (handle, 0) ;
} /* ver3_do_device */

void
handle_zytouch (Display *display, const CONFIG_COMMON * config)
{	struct sigaction sigact ;
	ZYTOUCH zyt ;

	if (config->daemon_mode)
		daemonize () ;

	/* Install SIGHUP handler so we can relaod the config. */
 	memset (&sigact, 0, sizeof (sigact)) ;
	sigact.sa_handler = sighup_handler ;
	sigaction (SIGHUP, &sigact, NULL) ;

	while (1)
	{	
		if (caught_sighup == 0)
			if (find_handle_retry (&zyt))
			{	sleep (5) ;
				continue ;
				} ;

		caught_sighup = 0 ;

		if (zyt.handle == NULL)
		{	sleep (5) ;
			continue ;
			} ;

		switch (zyt.product)
		{	case 2 :
				ver2_do_device (zyt.handle, display, config) ;
				break ;

			case 3 :
				ver3_do_device (zyt.handle, display, config) ;
				break ;

			default :
				printf ("Zytouch device %04x not supported yet.\n", zyt.product) ;
				break ;
			} ;

		if (caught_sighup)
			/* Don't close, just continue. */
			continue ;

		usb_close (zyt.handle) ;
		zyt.handle = NULL ;

		sleep (1) ;
		} ;

	return ;
} /* handle_zytouch */


